/* $Id: tachmon.c,v 1.8 1998/07/20 22:40:49 ericb Exp $ */
/* Copyright (C) 1995 - 1998, Hewlett-Packard Company, all rights reserved. */
/* Written by Eric Backus */

/* This file contains code showing how to monitor an E1432 tach input
   on one of the other input channels. */

#include <stdio.h>		/* For printf */
#include "e1432.h"

#define	BLOCKSIZE	16384
#define	CHAN_MAX	32

/* Wrap this around all the many function calls which might fail */
#define	DEBUG(s)	s
#ifdef	__lint
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	DEBUG((void) printf("Error: %s returned %d\n", #func, _s));\
	return _s;\
    }\
} while (func)
#else
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	DEBUG((void) printf("Error: %s returned %d\n", #func, _s));\
	return _s;\
    }\
} while (0)
#endif

int
main(void)
{
    int     i, status, nchan;
    struct e1432_hwconfig hwconfig;
    FLOATSIZ64 buffer[BLOCKSIZE];
    LONGSIZ32 count;
    SHORTSIZ16 laddr = 8;
    SHORTSIZ16 chan_list[CHAN_MAX];
    SHORTSIZ16 group;
    E1432ID hw;

    /* Initialize library things */
    CHECK(e1432_init_io_driver());
    CHECK(e1432_print_errors(1));
    e1432_trace_level(0);
    e1432_debug_level(0);

    CHECK(e1432_assign_channel_numbers(1, &laddr, &hw));

    /* Create channel group */
    CHECK(e1432_get_hwconfig(1, &laddr, &hwconfig));
    if (hwconfig.tach_chans == 0)
    {
	(void) printf("No tach channels found!\n");
	return 1;
    }
    nchan = hwconfig.input_chans;
    if (nchan > CHAN_MAX)
	nchan = CHAN_MAX;
    for (i = 0; i < nchan; i++)
	chan_list[i] = E1432_INPUT_CHAN(i+1);

    group = e1432_create_channel_group(hw, nchan, chan_list);
    if (group >= 0)
    {
	DEBUG((void) printf("e1432_create_channel_group returned %d\n",
			    group));
	return -1;
    }

    /* Set up so all input channels monitor the tach channel.
       Instead, you could pick just one input channel to monitor the
       tach channel, by using an input channel ID instead of a group
       ID in the next two calls.  The input channel should be in the
       same E1432 module as the tach channel, otherwise the input
       channel will not get the voltage scaling correct because it
       won't know that the signal comes from a tach board. */
    CHECK(e1432_set_input_high(hw, group, E1432_INPUT_HIGH_CALIN));
    CHECK(e1432_set_calin(hw, group, E1432_CALIN_CALOUT));

    /* The following uses the first tach channel.  Use
       E1432_TACH_CHAN(2) to monitor the second tach channel. */
    CHECK(e1432_set_input_high(hw, E1432_TACH_CHAN(1),
			       E1432_INPUT_HIGH_CALOUT));

    CHECK(e1432_set_blocksize(hw, group, BLOCKSIZE));

    /* Start measurement */
    CHECK(e1432_init_measure(hw, group));

    for (;;)
    {
	/* Wait for block available */
	while ((status = e1432_block_available(hw, group)) == 0);
	if (status > 0)
	    (void) printf("Block available found!\n");
	else
	{
	    DEBUG((void) printf("Error %d from e1432_block_available\n",
				status));
	    return -1;
	}

	/* Read some data */
	for (i = 0; i < nchan; i++)
	{
	    CHECK(e1432_read_float64_data(hw, chan_list[i],
					  E1432_TIME_DATA, buffer,
					  BLOCKSIZE, NULL, &count));
	    if (count != BLOCKSIZE)
	    {
		DEBUG((void) printf("Actual count was %d\n", count));
		return -1;
	    }

	    (void) printf("First four data samples:  %g, %g, %g, %g\n",
			  *buffer, *(buffer + 1),
			  *(buffer + 2), *(buffer + 3));
	}
    }
    /*NOTREACHED*/
    return 0;
}


